<?php

namespace App\Filament\Resources\Complaints\Tables;

use Filament\Actions\Action;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Support\Facades\DB;

class ComplaintsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('title')
                    ->label('Judul')
                    ->description(fn($record) => $record->resident->name)
                    ->searchable()
                    ->sortable(),

                TextColumn::make('status_label')
                    ->label('Status')
                    ->badge()
                    ->colors([
                        'gray'    => 'MENUNGGU',
                        'warning' => 'DIPROSES',
                        'success' => 'SELESAI',
                        'danger'  => 'DITOLAK',
                    ])
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('Dibuat')
                    ->dateTime('d M Y H:i')
                    ->sortable(),
            ])
            ->filters([
                TrashedFilter::make(),
            ])
            ->recordActions([
                ViewAction::make(),
                // 🔸 Diproses (hanya muncul jika status = 0 / Menunggu)
                Action::make('diproses')
                    ->label('Proses')
                    ->icon('heroicon-m-arrow-path')
                    ->color('warning')
                    ->visible(fn($record) => $record->status === 0)
                    ->schema([
                        Textarea::make('note')
                            ->label('Catatan')
                            ->placeholder('Contoh: Aduan sedang diverifikasi oleh perangkat desa')
                            ->required(),
                    ])
                    ->action(function ($record, $data) {
                        DB::transaction(function () use ($record, $data) {
                            $record->update(['status' => 1]);
                            $record->addHistory(1, $data['note']);
                        });
                    }),

                // 🔸 Selesai (hanya muncul jika status = 1 / Diproses)
                Action::make('selesai')
                    ->label('Selesai')
                    ->icon('heroicon-m-check-circle')
                    ->color('success')
                    ->visible(fn($record) => $record->status === 1)
                    ->schema([
                        Textarea::make('note')
                            ->label('Catatan')
                            ->placeholder('Contoh: Aduan telah ditindaklanjuti dan selesai ditangani')
                            ->required(),
                    ])
                    ->action(function ($record, $data) {
                        DB::transaction(function () use ($record, $data) {
                            $record->update(['status' => 2]);
                            $record->addHistory(2, $data['note']);
                        });
                    }),

                // 🔸 Ditolak (hanya muncul jika status = 0 / Menunggu)
                Action::make('ditolak')
                    ->label('Tolak')
                    ->icon('heroicon-m-x-circle')
                    ->color('danger')
                    ->visible(fn($record) => $record->status === 0)
                    ->schema([
                        Textarea::make('note')
                            ->label('Alasan Penolakan')
                            ->placeholder('Alasan mengapa aduan ditolak')
                            ->required(),
                    ])
                    ->action(function ($record, $data) {
                        DB::transaction(function () use ($record, $data) {
                            $record->update(['status' => 3]);
                            $record->addHistory(3, $data['note']);
                        });
                    }),
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make()
                ])
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ]);
    }
}
