<?php

namespace App\Filament\Widgets;

use App\Models\Resident;
use Carbon\Carbon;
use Filament\Support\RawJs;
use Filament\Widgets\ChartWidget;

class AgeGenderChart extends ChartWidget
{
    protected static ?int $sort = 2;
    protected int | string | array $columnSpan = 'full';
    protected ?string $maxHeight = '250px';

    protected function getData(): array
    {
        $now = Carbon::now();

        // Definisi kategori umur
        $categories = [
            'Bayi (<1)'     => [0, 0],
            'Balita (1-5)'  => [1, 5],
            'Anak (6-12)'   => [6, 12],
            'Remaja (13-17)' => [13, 17],
            'Dewasa (18-59)' => [18, 59],
            'Lansia (60+)'   => [60, 200], // 200 biar pasti cover semua
        ];

        $maleData = [];
        $femaleData = [];

        foreach ($categories as $label => [$min, $max]) {
            $male = Resident::whereHas('gender', fn($q) => $q->where('name', 'Laki-laki'))
                ->whereBetween('birth_date', [
                    $now->copy()->subYears($max)->startOfYear(),
                    $now->copy()->subYears($min)->endOfYear()
                ])->count();

            $female = Resident::whereHas('gender', fn($q) => $q->where('name', 'Perempuan'))
                ->whereBetween('birth_date', [
                    $now->copy()->subYears($max)->startOfYear(),
                    $now->copy()->subYears($min)->endOfYear()
                ])->count();

            // Untuk piramida penduduk, data laki-laki dibuat negatif supaya muncul di sisi kiri
            $maleData[] = -$male;
            $femaleData[] = $female;
        }

        return [
            'datasets' => [
                [
                    'label' => 'Laki-laki',
                    'data' => $maleData,
                    'backgroundColor' => '#36A2EB',
                    'borderWidth' => 0,
                ],
                [
                    'label' => 'Perempuan',
                    'data' => $femaleData,
                    'backgroundColor' => '#FF6384',
                    'borderWidth' => 0,
                ],
            ],
            'labels' => array_keys($categories),
        ];
    }

    protected function getType(): string
    {
        return 'bar';
    }

    protected function getOptions(): RawJs
    {
        return RawJs::make(<<<'JS'
        {
            indexAxis: 'y',
            responsive: true,
            maintainAspectRatio: true,
            scales: {
                x: {
                    stacked: true,
                    ticks: {
                        callback: (value) => Math.abs(value),
                    },
                },
                y: {
                    stacked: true,
                },
            },
            plugins: {
                datalabels: {
                    display: false,
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            const datasetLabel = context.dataset.label;
                            const value = Math.abs(context.parsed.x);

                            // Hitung total per kategori (Laki-laki + Perempuan)
                            const total = context.chart.data.datasets.reduce((sum, ds) => {
                                return sum + Math.abs(ds.data[context.dataIndex]);
                            }, 0);

                            const percentage = ((value / total) * 100).toFixed(1);

                            return datasetLabel + ': ' + value + ' (' + percentage + '%)';
                        },
                    },
                },
                legend: {
                    position: 'top',
                    labels: {
                        generateLabels: function(chart) {
                            // Hitung total keseluruhan semua dataset
                            const grandTotal = chart.data.datasets.reduce((sum, ds) => {
                                return sum + ds.data.reduce((s, val) => s + Math.abs(val), 0);
                            }, 0);

                            return chart.data.datasets.map(function(dataset, i) {
                                // Total dataset saat ini
                                const datasetTotal = dataset.data.reduce((sum, val) => sum + Math.abs(val), 0);
                                const percentage = ((datasetTotal / grandTotal) * 100).toFixed(1);

                                return {
                                    text: dataset.label + ': ' + datasetTotal + ' (' + percentage + '%)',
                                    fillStyle: dataset.backgroundColor,
                                    strokeStyle: dataset.borderColor,
                                    lineWidth: dataset.borderWidth,
                                    hidden: !chart.isDatasetVisible(i),
                                    index: i,
                                    fontColor: '#71717b',
                                };
                            });
                        }
                    }
                }
            },
        }
        JS);
    }
}
