<?php

namespace App\Filament\Widgets;

use App\Models\Resident;
use Filament\Support\RawJs;
use Filament\Widgets\ChartWidget;

class ReligionChart extends ChartWidget
{
    protected static ?int $sort = 7;
    protected ?string $maxHeight = '250px';

    protected function getData(): array
    {
        $data = Resident::with('religion')
            ->selectRaw('religion_id, COUNT(*) as total')
            ->groupBy('religion_id')
            ->get();

        $labels = $data->map(fn($item) => $item->religion?->name ?? 'Unknown');
        $totals = $data->pluck('total');

        return [
            'datasets' => [
                [
                    'data' => $totals,
                    'backgroundColor' => [
                        '#60A5FA',
                        '#34D399',
                        '#FBBF24',
                        '#F87171',
                        '#A78BFA',
                        '#F472B6',
                        '#94A3B8',
                    ],
                ],
            ],
            'labels' => $labels,
        ];
    }

    protected function getType(): string
    {
        return 'doughnut';
    }

    protected function getOptions(): RawJs
    {
        return RawJs::make(<<<'JS'
        {
            plugins: {
                legend: {
                    onClick: null,
                    position: 'right',
                    labels: {
                        generateLabels: function(chart) {
                            const data = chart.data;
                            if (data.labels.length && data.datasets.length) {
                                const dataset = data.datasets[0];
                                const total = dataset.data.reduce((a, b) => a + b, 0);

                                return data.labels.map((label, i) => {
                                    const value = dataset.data[i];
                                    const percentage = ((value / total) * 100).toFixed(1) + '%';
                                    const background = dataset.backgroundColor[i];

                                    return {
                                        text: label + ' : ' + value + ' (' + percentage + ')',
                                        fillStyle: background,
                                        hidden: isNaN(value) || dataset.data[i] === null,
                                        index: i,
                                        fontColor: '#71717b',
                                    };
                                });
                            }
                            return [];
                        }
                    }
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            let label = context.label || '';
                            let value = context.raw || 0;
                            let dataset = context.chart.data.datasets[0].data;
                            let total = dataset.reduce((a, b) => a + b, 0);
                            let percentage = ((value / total) * 100).toFixed(1) + '%';
                            return label + ': ' + value + ' (' + percentage + ')';
                        }
                    }
                }
            }
        }
        JS);
    }
}
